//==============================================================================
// Project:		Wealth Transfers and their Economic Effects
// File name:	SIH Gift Under-reporting
// Objective: 	Extracting gift inflow and outflow variables from every
//				iteration of the SIH to provide insight into the level of
//				under-reporting and the growth over time.
//
// Created: 	16/07/2021
// Modified: 	15/11/2021
//==============================================================================

********************************
*** Data import and cleaning ***
********************************

clear

cd "XXXX"	// set the content directory here

use sih07bp.dta	// importing the data

keep SIHPSWT PFYIFAM PFYSFAM CWFINSPP CWIFNIH QUARTERH	// keeping relevant variables

gen year = 2007	// generating a year variables

save SIH_07.dta, replace

clear

use sih09bp.dta

keep SIHPSWT PFYIFAM PFYSFAM CWFINSPP CWIFNIH QUARTERH

gen year = 2009

save SIH_09.dta, replace

clear

use sih11bp.dta

keep SIHPSWT PFYIFAM PFYSFAM CWFINSPP CWIFNIH QUARTERH

gen year = 2011

save SIH_11.dta, replace

clear

use sih13bp.dta

keep SIHPSWT PFYIFAM PFYSFAM CWFINSPP CWIFNIH QUARTERH

gen year = 2013

save SIH_13.dta, replace

clear

use sih15bp.dta

keep sihpswt pfyifam pfysfam cwfinspp cwifnih qtritrwh

rename sihpswt SIHPSWT
rename pfyifam PFYIFAM
rename pfysfam PFYSFAM
rename cwfinspp CWFINSPP
rename cwifnih CWIFNIH
rename qtritrwh QUARTERH

gen year = 2015

save SIH_15.dta, replace

clear

use sih17bp.dta

keep sihpswt cwfinspp cwifnih qtritrwh

rename sihpswt SIHPSWT
rename cwfinspp CWFINSPP
rename cwifnih CWIFNIH
rename qtritrwh QUARTERH

gen year = 2017

save SIH_17.dta, replace

clear

use SIH_07
append using SIH_09
append using SIH_11
append using SIH_13
append using SIH_15
append using SIH_17

*********************
*** Data analysis ***
*********************

by QUARTERH year, sort: egen num_interviews = count(QUARTERH)	// counting the number of interviews per quarter
by year, sort: egen total_interviews = count(QUARTERH)	// counting the total number of interviews
by QUARTERH year, sort: gen qtr_wt = total_interviews/(num_interviews * 4)	// creating an interview quarter weight so that each quarter is equally represented

//////////////////////////////
/// Current financial year ///
//////////////////////////////

* generating current financial year weights
gen cfy_wt = SIHPSWT * qtr_wt

* converting weekly values into annual values
local cfy_vars CWIFNIH CWFINSPP
foreach v in `cfy_vars' {
	replace `v' = `v' * 52
}

* mean current financial year inflow
gen mean_cfy_inflow = .
foreach x of numlist 2007(2)2017 {
	sum CWIFNIH [w = cfy_wt] if CWIFNIH > 0 & year == `x', detail
	gen mean_cfy_inflow_`x' = r(mean)
	replace mean_cfy_inflow = mean_cfy_inflow_`x' if year == `x'
	drop mean_cfy_inflow_`x'
}

* sum current financial year inflow
gen pop_CWIFNIH = CWIFNIH * cfy_wt
by year, sort: egen sum_cfy_inflow = sum(pop_CWIFNIH)

* count current financial year inflow
by year, sort: egen count_cfy_inflow = sum(cfy_wt) if CWIFNIH > 0

* mean current financial year outflow
gen mean_cfy_outflow = .
foreach x of numlist 2007(2)2017 {
	sum CWFINSPP [w = cfy_wt] if CWFINSPP > 0 & year == `x', detail
	gen mean_cfy_outflow_`x' = r(mean)
	replace mean_cfy_outflow = mean_cfy_outflow_`x' if year == `x'
	drop mean_cfy_outflow_`x'
}

* sum current financial year outflow
gen pop_CWFINSPP = CWFINSPP * cfy_wt
by year, sort: egen sum_cfy_outflow = sum(pop_CWFINSPP)

* count current financial year outflow
by year, sort: egen count_cfy_outflow = sum(cfy_wt) if CWFINSPP > 0

///////////////////////////////
/// Previous financial year ///
///////////////////////////////

* mean previous financial year inflow
gen mean_pfy_inflow = .
foreach x of numlist 2007(2)2017 {
	sum PFYIFAM [w = SIHPSWT] if PFYIFAM > 0 & year == `x', detail
	gen mean_pfy_inflow_`x' = r(mean)
	replace mean_pfy_inflow = mean_pfy_inflow_`x' if year == `x'
	drop mean_pfy_inflow_`x'
}

* sum previous financial year inflow
gen pop_PFYIFAM = PFYIFAM * SIHPSWT
by year, sort: egen sum_pfy_inflow = sum(pop_PFYIFAM)

* count previous financial year inflow
by year, sort: egen count_pfy_inflow = sum(SIHPSWT) if PFYIFAM > 0

* mean previous financial year outflow
gen mean_pfy_outflow = .
foreach x of numlist 2007(2)2017 {
	sum PFYSFAM [w = SIHPSWT] if PFYSFAM > 0 & year == `x', detail
	gen mean_pfy_outflow_`x' = r(mean)
	replace mean_pfy_outflow = mean_pfy_outflow_`x' if year == `x'
	drop mean_pfy_outflow_`x'
}

* sum previous financial year outflow
gen pop_PFYSFAM = PFYSFAM * SIHPSWT
by year, sort: egen sum_pfy_outflow = sum(pop_PFYSFAM)

* count previous financial year outflow
by year, sort: egen count_pfy_outflow = sum(SIHPSWT) if PFYSFAM > 0

collapse(mean) sum* mean* count*, by(year)

replace count_pfy_inflow = . if year == 2017
replace count_pfy_outflow = . if year == 2017
replace sum_pfy_inflow = . if year == 2017
replace sum_pfy_outflow = . if year == 2017

save "SIH_gifts.dta", replace	// saving the dataset